/*
 * Decompiled with CFR 0.152.
 */
package me.redstoner2019.streamingclient.capture;

import java.awt.AWTException;
import java.awt.GraphicsDevice;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.net.URI;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import javax.sound.sampled.Mixer;
import me.redstoner2019.streamingclient.capture.AudioCapture;
import me.redstoner2019.streamingclient.capture.DesktopAudioCapture;
import me.redstoner2019.streamingclient.capture.QualityPreset;
import me.redstoner2019.streamingclient.capture.ScreenCapture;
import me.redstoner2019.streamingclient.network.StreamWebSocketClient;
import me.redstoner2019.streamingclient.recording.VideoRecorder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StreamingService {
    private static final Logger logger = LoggerFactory.getLogger(StreamingService.class);
    private ScreenCapture screenCapture;
    private AudioCapture audioCapture;
    private DesktopAudioCapture desktopAudioCapture;
    private StreamWebSocketClient webSocketClient;
    private VideoRecorder videoRecorder;
    private ScheduledExecutorService executor;
    private ScheduledFuture<?> captureTask;
    private final AtomicBoolean streaming = new AtomicBoolean(false);
    private final AtomicBoolean recording = new AtomicBoolean(false);
    private final AtomicBoolean audioEnabled = new AtomicBoolean(true);
    private final AtomicLong framesSent = new AtomicLong(0L);
    private final AtomicLong framesDropped = new AtomicLong(0L);
    private final AtomicLong audioPacketsSent = new AtomicLong(0L);
    private String serverUrl;
    private String streamId;
    private QualityPreset qualityPreset = QualityPreset.HIGH;
    private ScreenCapture.CompressionMode compressionMode = ScreenCapture.CompressionMode.MEDIUM;
    private int customWidth = 1920;
    private int customHeight = 1080;
    private int customFps = 30;
    private Consumer<String> statusListener;
    private Consumer<Long> frameCountListener;
    private Consumer<Exception> errorListener;
    private Consumer<BufferedImage> previewListener;

    public StreamingService() {
        this.videoRecorder = new VideoRecorder();
    }

    public void connect(String serverUrl, String streamId) throws Exception {
        this.serverUrl = serverUrl;
        this.streamId = streamId;
        Object wsUrl = serverUrl.replace("http://", "ws://").replace("https://", "wss://");
        if (!((String)wsUrl).endsWith("/stream")) {
            wsUrl = (String)wsUrl + "/stream";
        }
        this.webSocketClient = new StreamWebSocketClient(new URI((String)wsUrl));
        this.webSocketClient.setOnConnected(() -> {
            this.webSocketClient.registerAsStreamer(streamId);
            this.notifyStatus("Connected and registered as streamer: " + streamId);
        });
        this.webSocketClient.setOnTextMessage(message -> {
            logger.debug("Server message: {}", message);
            if (message.startsWith("STREAMING:")) {
                this.notifyStatus("Server confirmed streaming");
            }
        });
        this.webSocketClient.setOnDisconnected(reason -> {
            this.notifyStatus("Disconnected: " + reason);
            this.stopStreaming();
        });
        this.webSocketClient.setOnError(e -> {
            if (this.errorListener != null) {
                this.errorListener.accept((Exception)e);
            }
        });
        this.webSocketClient.connectBlocking(5L, TimeUnit.SECONDS);
        if (!this.webSocketClient.isOpen()) {
            throw new Exception("Failed to connect to server");
        }
    }

    public void startStreaming() throws AWTException {
        this.startStreaming(null, false);
    }

    public void startStreaming(Mixer.Info audioMixer) throws AWTException {
        this.startStreaming(audioMixer, false);
    }

    public void startStreaming(Mixer.Info audioMixer, boolean useDesktopAudio) throws AWTException {
        int fps;
        int height;
        int width;
        if (this.streaming.get()) {
            return;
        }
        this.screenCapture = new ScreenCapture();
        this.screenCapture.setCompressionMode(this.compressionMode);
        if (this.qualityPreset == QualityPreset.CUSTOM) {
            width = this.customWidth;
            height = this.customHeight;
            fps = this.customFps;
        } else {
            width = this.qualityPreset.getWidth();
            height = this.qualityPreset.getHeight();
            fps = this.qualityPreset.getFps();
        }
        this.screenCapture.setTargetResolution(width, height);
        if (this.audioEnabled.get()) {
            try {
                if (useDesktopAudio) {
                    this.desktopAudioCapture = new DesktopAudioCapture();
                    this.desktopAudioCapture.setAudioDataListener(this::handleAudioData);
                    this.desktopAudioCapture.startCapture();
                    logger.info("Desktop audio capture started (WASAPI loopback)");
                } else {
                    this.audioCapture = new AudioCapture();
                    this.audioCapture.setAudioDataListener(this::handleAudioData);
                    if (audioMixer != null) {
                        this.audioCapture.startCapture(audioMixer);
                        logger.info("Audio capture started with device: {}", (Object)audioMixer.getName());
                    } else {
                        this.audioCapture.startCapture();
                        logger.info("Audio capture started with default device");
                    }
                }
                this.audioPacketsSent.set(0L);
            }
            catch (Exception e) {
                logger.warn("Failed to start audio capture: {}", (Object)e.getMessage());
                this.notifyStatus("Warning: Audio capture unavailable - " + e.getMessage());
                this.audioCapture = null;
                this.desktopAudioCapture = null;
            }
        }
        this.executor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "ScreenCapture-Thread");
            t.setDaemon(true);
            return t;
        });
        long frameInterval = 1000 / fps;
        this.streaming.set(true);
        this.framesSent.set(0L);
        this.framesDropped.set(0L);
        this.captureTask = this.executor.scheduleAtFixedRate(this::captureAndSendFrame, 0L, frameInterval, TimeUnit.MILLISECONDS);
        String audioStatus = this.audioCapture != null ? " with audio" : "";
        this.notifyStatus("Streaming started at " + width + "x" + height + " @ " + fps + "fps" + audioStatus);
        logger.info("Started streaming: {}x{} @ {}fps, compression: {}, audio: {}", new Object[]{width, height, fps, this.compressionMode, this.audioCapture != null});
    }

    public void stopStreaming() {
        if (!this.streaming.get()) {
            return;
        }
        this.streaming.set(false);
        if (this.captureTask != null) {
            this.captureTask.cancel(false);
            this.captureTask = null;
        }
        if (this.executor != null) {
            this.executor.shutdown();
            try {
                this.executor.awaitTermination(2L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            this.executor = null;
        }
        if (this.audioCapture != null) {
            this.audioCapture.stopCapture();
            this.audioCapture = null;
        }
        if (this.desktopAudioCapture != null) {
            this.desktopAudioCapture.stopCapture();
            this.desktopAudioCapture = null;
        }
        this.notifyStatus("Streaming stopped");
        logger.info("Streaming stopped. Frames sent: {}, Frames dropped: {}, Audio packets sent: {}", this.framesSent.get(), this.framesDropped.get(), this.audioPacketsSent.get());
    }

    public void disconnect() {
        this.stopStreaming();
        this.stopRecording();
        if (this.webSocketClient != null && this.webSocketClient.isOpen()) {
            this.webSocketClient.close();
            this.webSocketClient = null;
        }
    }

    public String startRecording(String outputDir, VideoRecorder.VideoFormat format) throws Exception {
        int fps;
        int height;
        int width;
        if (this.recording.get()) {
            throw new IllegalStateException("Already recording");
        }
        if (this.qualityPreset == QualityPreset.CUSTOM) {
            width = this.customWidth;
            height = this.customHeight;
            fps = this.customFps;
        } else {
            width = this.qualityPreset.getWidth();
            height = this.qualityPreset.getHeight();
            fps = this.qualityPreset.getFps();
        }
        int bitrate = width * height * fps / 10;
        String path = this.videoRecorder.startRecording(outputDir, width, height, fps, bitrate, format);
        this.recording.set(true);
        this.notifyStatus("Recording started: " + path);
        return path;
    }

    public void stopRecording() {
        if (!this.recording.get()) {
            return;
        }
        this.videoRecorder.stopRecording();
        this.recording.set(false);
        this.notifyStatus("Recording stopped");
    }

    private void captureAndSendFrame() {
        if (!this.streaming.get()) {
            return;
        }
        try {
            BufferedImage frame2 = this.screenCapture.captureScreen();
            if (this.previewListener != null) {
                this.previewListener.accept(frame2);
            }
            if (this.recording.get()) {
                this.videoRecorder.addFrame(frame2);
            }
            byte[] encoded = this.screenCapture.encodeImage(frame2);
            if (this.webSocketClient != null && this.webSocketClient.isOpen()) {
                this.webSocketClient.sendFrame(encoded);
                this.framesSent.incrementAndGet();
                if (this.frameCountListener != null) {
                    this.frameCountListener.accept(this.framesSent.get());
                }
            }
        }
        catch (IOException e) {
            this.framesDropped.incrementAndGet();
            logger.warn("Failed to capture/send frame: {}", (Object)e.getMessage());
        }
    }

    private void handleAudioData(byte[] audioData, int length) {
        if (!this.streaming.get() || this.webSocketClient == null || !this.webSocketClient.isOpen()) {
            return;
        }
        try {
            byte[] dataToSend = new byte[length];
            System.arraycopy(audioData, 0, dataToSend, 0, length);
            this.webSocketClient.sendFrame(dataToSend, StreamWebSocketClient.FrameType.AUDIO);
            this.audioPacketsSent.incrementAndGet();
        }
        catch (Exception e) {
            logger.warn("Failed to send audio data: {}", (Object)e.getMessage());
        }
    }

    private void notifyStatus(String status) {
        if (this.statusListener != null) {
            this.statusListener.accept(status);
        }
    }

    public void setQualityPreset(QualityPreset preset) {
        this.qualityPreset = preset;
    }

    public void setCompressionMode(ScreenCapture.CompressionMode mode) {
        this.compressionMode = mode;
    }

    public void setCustomResolution(int width, int height, int fps) {
        this.customWidth = width;
        this.customHeight = height;
        this.customFps = fps;
        this.qualityPreset = QualityPreset.CUSTOM;
    }

    public void setScreen(GraphicsDevice device2) throws AWTException {
        if (this.screenCapture != null) {
            this.screenCapture.setDevice(device2);
        }
    }

    public void setStatusListener(Consumer<String> listener) {
        this.statusListener = listener;
    }

    public void setFrameCountListener(Consumer<Long> listener) {
        this.frameCountListener = listener;
    }

    public void setErrorListener(Consumer<Exception> listener) {
        this.errorListener = listener;
    }

    public void setPreviewListener(Consumer<BufferedImage> listener) {
        this.previewListener = listener;
    }

    public boolean isStreaming() {
        return this.streaming.get();
    }

    public boolean isRecording() {
        return this.recording.get();
    }

    public boolean isConnected() {
        return this.webSocketClient != null && this.webSocketClient.isOpen();
    }

    public long getFramesSent() {
        return this.framesSent.get();
    }

    public long getFramesDropped() {
        return this.framesDropped.get();
    }

    public long getBytesSent() {
        return this.webSocketClient != null ? this.webSocketClient.getBytesSent() : 0L;
    }

    public String getStreamId() {
        return this.streamId;
    }

    public VideoRecorder getVideoRecorder() {
        return this.videoRecorder;
    }

    public void setAudioEnabled(boolean enabled) {
        this.audioEnabled.set(enabled);
    }

    public boolean isAudioEnabled() {
        return this.audioEnabled.get();
    }

    public long getAudioPacketsSent() {
        return this.audioPacketsSent.get();
    }
}

