/*
 * Decompiled with CFR 0.152.
 */
package me.redstoner2019.streamingclient.capture;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.net.URI;
import java.nio.ByteBuffer;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import javax.imageio.ImageIO;
import me.redstoner2019.streamingclient.network.StreamWebSocketClient;
import me.redstoner2019.streamingclient.recording.VideoRecorder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ViewingService {
    private static final Logger logger = LoggerFactory.getLogger(ViewingService.class);
    private StreamWebSocketClient webSocketClient;
    private VideoRecorder videoRecorder;
    private final AtomicBoolean watching = new AtomicBoolean(false);
    private final AtomicBoolean recording = new AtomicBoolean(false);
    private final AtomicLong framesReceived = new AtomicLong(0L);
    private final AtomicLong framesDropped = new AtomicLong(0L);
    private String serverUrl;
    private String streamId;
    private BufferedImage latestFrame;
    private int frameWidth = 1920;
    private int frameHeight = 1080;
    private int estimatedFps = 30;
    private Consumer<String> statusListener;
    private Consumer<BufferedImage> frameListener;
    private Consumer<Exception> errorListener;
    private Consumer<Long> frameCountListener;

    public ViewingService() {
        this.videoRecorder = new VideoRecorder();
    }

    public void connect(String serverUrl, String streamId) throws Exception {
        this.serverUrl = serverUrl;
        this.streamId = streamId;
        Object wsUrl = serverUrl.replace("http://", "ws://").replace("https://", "wss://");
        if (!((String)wsUrl).endsWith("/stream")) {
            wsUrl = (String)wsUrl + "/stream";
        }
        this.webSocketClient = new StreamWebSocketClient(new URI((String)wsUrl));
        this.webSocketClient.setOnConnected(() -> {
            this.webSocketClient.registerAsViewer(streamId);
            this.notifyStatus("Connected and watching stream: " + streamId);
            this.watching.set(true);
        });
        this.webSocketClient.setOnTextMessage(message -> {
            logger.debug("Server message: {}", message);
            if (message.startsWith("WATCHING:")) {
                this.notifyStatus("Server confirmed watching");
            }
        });
        this.webSocketClient.setOnBinaryMessage(this::handleFrame);
        this.webSocketClient.setOnDisconnected(reason -> {
            this.notifyStatus("Disconnected: " + reason);
            this.watching.set(false);
        });
        this.webSocketClient.setOnError(e -> {
            if (this.errorListener != null) {
                this.errorListener.accept((Exception)e);
            }
        });
        this.webSocketClient.connectBlocking(5L, TimeUnit.SECONDS);
        if (!this.webSocketClient.isOpen()) {
            throw new Exception("Failed to connect to server");
        }
    }

    private void handleFrame(ByteBuffer data) {
        try {
            byte[] bytes = new byte[data.remaining()];
            data.get(bytes);
            BufferedImage frame2 = ImageIO.read(new ByteArrayInputStream(bytes));
            if (frame2 != null) {
                if (frame2.getWidth() != this.frameWidth || frame2.getHeight() != this.frameHeight) {
                    this.frameWidth = frame2.getWidth();
                    this.frameHeight = frame2.getHeight();
                    logger.info("Stream resolution: {}x{}", (Object)this.frameWidth, (Object)this.frameHeight);
                }
                this.latestFrame = frame2;
                this.framesReceived.incrementAndGet();
                if (this.frameListener != null) {
                    this.frameListener.accept(frame2);
                }
                if (this.recording.get()) {
                    this.videoRecorder.addFrame(frame2);
                }
                if (this.frameCountListener != null) {
                    this.frameCountListener.accept(this.framesReceived.get());
                }
            } else {
                this.framesDropped.incrementAndGet();
            }
        }
        catch (Exception e) {
            this.framesDropped.incrementAndGet();
            logger.warn("Failed to decode frame: {}", (Object)e.getMessage());
        }
    }

    public void disconnect() {
        this.watching.set(false);
        this.stopRecording();
        if (this.webSocketClient != null && this.webSocketClient.isOpen()) {
            this.webSocketClient.close();
            this.webSocketClient = null;
        }
        this.notifyStatus("Disconnected from stream");
    }

    public String startRecording(String outputDir, VideoRecorder.VideoFormat format) throws Exception {
        if (this.recording.get()) {
            throw new IllegalStateException("Already recording");
        }
        int bitrate = this.frameWidth * this.frameHeight * this.estimatedFps / 10;
        String path = this.videoRecorder.startRecording(outputDir, this.frameWidth, this.frameHeight, this.estimatedFps, bitrate, format);
        this.recording.set(true);
        this.notifyStatus("Recording started: " + path);
        return path;
    }

    public void stopRecording() {
        if (!this.recording.get()) {
            return;
        }
        this.videoRecorder.stopRecording();
        this.recording.set(false);
        this.notifyStatus("Recording stopped: " + this.videoRecorder.getOutputPath());
    }

    private void notifyStatus(String status) {
        if (this.statusListener != null) {
            this.statusListener.accept(status);
        }
    }

    public void setStatusListener(Consumer<String> listener) {
        this.statusListener = listener;
    }

    public void setFrameListener(Consumer<BufferedImage> listener) {
        this.frameListener = listener;
    }

    public void setErrorListener(Consumer<Exception> listener) {
        this.errorListener = listener;
    }

    public void setFrameCountListener(Consumer<Long> listener) {
        this.frameCountListener = listener;
    }

    public boolean isWatching() {
        return this.watching.get();
    }

    public boolean isRecording() {
        return this.recording.get();
    }

    public boolean isConnected() {
        return this.webSocketClient != null && this.webSocketClient.isOpen();
    }

    public long getFramesReceived() {
        return this.framesReceived.get();
    }

    public long getFramesDropped() {
        return this.framesDropped.get();
    }

    public long getBytesReceived() {
        return this.webSocketClient != null ? this.webSocketClient.getBytesReceived() : 0L;
    }

    public BufferedImage getLatestFrame() {
        return this.latestFrame;
    }

    public String getStreamId() {
        return this.streamId;
    }

    public int getFrameWidth() {
        return this.frameWidth;
    }

    public int getFrameHeight() {
        return this.frameHeight;
    }

    public VideoRecorder getVideoRecorder() {
        return this.videoRecorder;
    }

    public void setEstimatedFps(int fps) {
        this.estimatedFps = fps;
    }
}

